<?php
/**
 * Email Test
 *
 * Tests PHP's ability to send mail using the native mail() function.
 *
 * @author     David Cassidy <david.cassidy@hostek.com>
 * @copyright  2024 Hostek
 * @license    https://unlicense.org/ Unlicense
 */

// Set default timezone
date_default_timezone_set('UTC');

// Set message fields
$msgTimestamp = date('m/d/Y H:i:s e', $_SERVER['REQUEST_TIME']);
$msgRecipient = !empty($_POST) && !empty($_POST['msgRecipient']) ? $_POST['msgRecipient'] : 'support@hostek.com';
$msgSender    = !empty($_POST) && !empty($_POST['msgSender'])    ? $_POST['msgSender']    : "server@${_SERVER['SERVER_NAME']}";
$msgReplyTo   = !empty($_POST) && !empty($_POST['msgReplyTo'])   ? $_POST['msgReplyTo']   : 'support@hostek.com';
$msgSubject   = !empty($_POST) && !empty($_POST['msgSubject'])   ? $_POST['msgSubject']   : "PHP Email Test - ${_SERVER['SERVER_NAME']}";
$msgContent   = <<<EOM
Hello, there!

This message was sent in an effort to test the PHP mail() functionality of the server from which it originated.

If you received this message, mail originating from the script referenced below is being sent and received successfully.

Please see the below diagnostic information for details.

----------[ Diagnostic Information ]----------

Timestamp:
${msgTimestamp}

Server:
${_SERVER['SERVER_NAME']}

Host:
${_SERVER['HTTP_HOST']}

Script:
${_SERVER['SCRIPT_FILENAME']}

SMTP Host:
${!${''} = ini_get('SMTP')}

SMTP Port:
${!${''} = ini_get('smtp_port')}

IP Address:
${_SERVER['REMOTE_ADDR']}

User-Agent:
${_SERVER['HTTP_USER_AGENT']}

----------------------------------------------

If you have questions or need technical assistance, please contact our support team.

Regards,
Hostek Support Team
EOM;

// Check form submission
if ($_POST) {

    // Initialize errors
    $formErrors = [];

    // Initialize status
    $formStatus = false;

    try {
        // Security check
        if (empty($_SERVER['HTTP_REFERER'])) {
            throw new Exception('Script cannot be called directly.');
            die();
        }

        // Validate fields
        if (!filter_var($msgSender, FILTER_VALIDATE_EMAIL)) {
            $formErrors[] = 'The FROM field must contain a valid email address.';
        }
        if (!filter_var($msgReplyTo, FILTER_VALIDATE_EMAIL)) {
            $formErrors[] = 'The REPLYTO field must contain a valid email address.';
        }
        if (!filter_var($msgRecipient, FILTER_VALIDATE_EMAIL)) {
            $formErrors[] = 'The TO field must contain a valid email address.';
        }
        if (empty($_POST['msgSubject'])) {
            $formErrors[] = 'The SUBJECT field is required.';
        }
        if (empty($_POST['msgContent'])) {
            $formErrors[] = 'The MESSAGE field is required.';
        }

        // Initialize message
        $msgSender    = $_POST['msgSender'];
        $msgReplyTo   = $_POST['msgReplyTo'];
        $msgRecipient = $_POST['msgRecipient'];
        $msgSubject   = $_POST['msgSubject'];
        $msgContent   = $_POST['msgContent'];

        // Send message
        $result = mail(
            $msgRecipient,
            $msgSubject,
            $msgContent,
            "From: ${msgSender}" . "\r\n" . "Reply-To: ${msgReplyTo}" . "\r\n"
        );

        // Check status
        if (!$result) {
            throw new Exception('Mail could not be sent. Check error log for details.');
            die();
        } else {
            $formStatus = true;
        }
    } catch (Exception $e) {
        echo $e->getMessage();
        die();
    }
}
?><!DOCTYPE html>
<html lang="en" dir="ltr">

    <head>

        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">

        <title>PHP Email Test</title>

        <link rel="preconnect" href="https://fonts.googleapis.com">
        <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
        <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Roboto&display=swap">

        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/tailwindcss/2.2.19/tailwind.min.css">
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/tailwindcss/2.2.19/components.min.css">

        <style>
            * {
                font-family: Roboto, san-serif !important;
                font-weight: 400 !important;
            }
        </style>

    </head>

    <body class="bg-gray-100 text-gray-400 p-6">

        <main class="max-w-2xl mx-auto">

            <!-- Container -->
            <div class="bg-white rounded-lg p-8">

                <div class="flex flex-row flex-nowrap justify-between items-center mb-6">
                    <h1 class="flex-1 w-1/2 text-black text-2xl leading-10">
                        PHP Email Test
                    </h1>
                    <div class="flex-1 w-1/2 text-right">
                        <img src="data:image/png;base64, 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"
                            class="inline-block max-h-10 ml-auto w-auto" alt="Logo">
                    </div>
                </div>

                <?php if(!empty($formErrors)): ?>
                    <!-- Errors -->
                    <div class="p-4 bg-red-100 text-red-600 text-sm rounded-md mb-6">
                        <p class="mb-2">
                            Please correct the following errors and try again:
                        </p>
                        <ul class="list-disc list-inside">
                            <?php foreach($formErrors as $error): ?>
                                <li><?= $error; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if(!empty($formStatus)): ?>
                    <!-- Errors -->
                    <div class="p-4 bg-green-100 text-green-600 text-sm rounded-md mb-6">
                        <p>
                            Your message has been sent successfully!
                        </p>
                    </div>
                <?php endif; ?>

                <!-- Form -->
                <form action="ht_mail.php" method="POST">

                    <!-- Message Sender -->
                    <div class="mb-6">
                        <label for="msgSender" class="block text-sm font-medium text-gray-700">
                            From
                            <span class="text-red-500 font-semibold text-xs">*</span>
                        </label>
                        <div class="mt-1">
                            <input type="email" name="msgSender" id="msgSender" value="<?= $msgSender; ?>" required
                                class="bg-gray-100 text-black py-2 px-4 block w-full sm:text-sm border-gray-300
                                    rounded-md focus:ring-indigo-500 focus:border-indigo-500 transition ease-in-out
                                    duration-300">
                        </div>
                    </div>

                    <!-- Message ReplyTo -->
                    <div class="mb-6">
                        <label for="msgReplyTo" class="block text-sm font-medium text-gray-700">
                            Reply To
                            <span class="text-red-500 font-semibold text-xs">*</span>
                        </label>
                        <div class="mt-1">
                            <input type="email" name="msgReplyTo" id="msgReplyTo" value="<?= $msgReplyTo; ?>" required
                                class="bg-gray-100 text-black py-2 px-4 block w-full sm:text-sm border-gray-300
                                    rounded-md focus:ring-indigo-500 focus:border-indigo-500 transition ease-in-out
                                    duration-300">
                        </div>
                    </div>

                    <!-- Message Recipient -->
                    <div class="mb-6">
                        <label for="msgRecipient" class="block text-sm font-medium text-gray-700">
                            To
                            <span class="text-red-500 font-semibold text-xs">*</span>
                        </label>
                        <div class="mt-1">
                            <input type="email" name="msgRecipient" id="msgRecipient" value="<?= $msgRecipient; ?>" required
                                class="bg-gray-100 text-black py-2 px-4 block w-full sm:text-sm border-gray-300
                                    rounded-md focus:ring-indigo-500 focus:border-indigo-500 transition ease-in-out
                                    duration-300">
                        </div>
                    </div>

                    <!-- Message Subject -->
                    <div class="mb-6">
                        <label for="msgSubject" class="block text-sm font-medium text-gray-700">
                            Subject
                            <span class="text-red-500 font-semibold text-xs">*</span>
                        </label>
                        <div class="mt-1">
                            <input type="text" name="msgSubject" id="msgSubject" value="<?= $msgSubject; ?>" required
                                class="bg-gray-100 text-black py-2 px-4 block w-full sm:text-sm border-gray-300
                                    rounded-md focus:ring-indigo-500 focus:border-indigo-500 transition ease-in-out
                                    duration-300">
                        </div>
                    </div>

                    <!-- Message Content -->
                    <div class="mb-6">
                        <label for="msgContent" class="block text-sm font-medium text-gray-700">
                            Message
                            <span class="text-red-500 font-semibold text-xs">*</span>
                        </label>
                        <div class="mt-1">
                            <textarea name="msgContent" id="msgContent" rows="10" required
                                class="bg-gray-100 text-black py-2 px-4 block w-full sm:text-sm border-gray-300
                                    rounded-md focus:ring-indigo-500 focus:border-indigo-500 transition ease-in-out
                                    duration-300"><?= $msgContent; ?></textarea>
                        </div>
                    </div>

                    <div>
                        <button type="submit" class="inline-flex items-center px-3 py-2 border border-transparent
                            shadow-sm text-sm leading-4 font-medium rounded-md text-white bg-indigo-600
                            hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2
                            focus:ring-indigo-500">
                            <svg class="-ml-0.5 mr-2 h-4 w-4" xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                <path d="M10.894 2.553a1 1 0 00-1.788 0l-7 14a1 1 0 001.169 1.409l5-1.429A1 1 0 009 15.571V11a1 1 0 112 0v4.571a1 1 0 00.725.962l5 1.428a1 1 0 001.17-1.408l-7-14z" />
                            </svg>
                            Send Message
                        </button>
                    </div>

                </form>

            </div>

        </main>

    </body>

</html>
