### FUNCTIONS ##################################################################

    # Check elevation
    function Test-Administrator
    {
        $CurrentUser = New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())
        $CurrentUser.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
    }

    # Initialize path
    function Init-Path
    {
        param(
            [Parameter(Mandatory)][string]$Path
        )
        Write-Host "`nInitializing $($Path) directory"
        if (!(Test-Path -PathType Container $Path)) {
            New-Item -ItemType Directory -Force -Path $Path | Out-Null
        }
    }

    # Download file
    function Get-File
    {
        param(
            [Parameter(Mandatory)][string]$FileSource,
            [Parameter(Mandatory)][string]$FileDestination
        )
        Write-Host "`nDownloading to $($FileDestination)"
        Invoke-WebRequest -Uri $FileSource -UseBasicParsing -OutFile $FileDestination
        if (!(Test-Path $FileDestination)) {
            Write-Host "Downloading $FileDestination failed" -Foregroundcolor Red
            Exit
        } else {
            Write-Host "Download complete" -ForegroundColor Green
        }
    }

    # Extract file
    function Unzip-File
    {
        param($UnzipSource, $UnzipDestination)
        Write-Host "`nExtracting $($UnzipSource) to $($UnzipDestination)"
        7zip x $UnzipSource -y -o"$($UnzipDestination)" | Out-Null
        Write-Host "Extraction complete" -ForegroundColor Green
    }

    # Install file
    function Install-File
    {
        param($InstallerSource)
        try {
            Write-Host "`nInstalling $InstallerSource"
            $proc = Start-Process -FilePath $InstallerSource -ArgumentList "/quiet /norestart" -Wait -PassThru
            $proc.waitForExit()
            Write-Host "Installation complete" -ForegroundColor Green
        } catch [expression] {
            Write-Host "Installation failed" -ForegroundColor Red
        }
    }


### INITIALIZE #################################################################

    Clear-Host

    [System.Net.ServicePointManager]::SecurityProtocol = 'Tls,Tls11,Tls12';

    # Check privileges
    if (!(Test-Administrator)) {
        Write-Host "This script requires elevation (Run as Administrator)" -ForegroundColor Red
        Exit
    }

    # Bundle
    $FileBundleName   = 'cpit_bundle'
    $FileBundleSource = 'https://gitlab1.hostek.com/api/v4/projects/380/repository/archive.zip?private_token=glpat-MUj34UmiSWHbLXuQhPMK'

    # Alias paths
    $InstallDirectory   = "C:\installs"
    $BundleDirectory    = "$($InstallDirectory)\$($FileBundleName)"
    $DownloadDirectory  = "$($BundleDirectory).zip"
    $EncryptorDirectory = "$($BundleDirectory)\cpit_encrypt"
    $EncryptorPath      = "$($EncryptorDirectory)\configuration-encryptor.exe"

    $ProgramDataDirectory = "C:\ProgramData"
    $CpitDirectory        = "$($ProgramDataDirectory)\controlpanel-integration-tools"
    $CputDirectory        = "$($ProgramDataDirectory)\controlpanel-update-tools"
    $CertificatePath      = "Cert:\LocalMachine\Root"

    # Certificate Thumbprint
    $CertificateThumbprint = '7287DAC177BFBA6D5A6DC12CC1A23BAE8138A26C'

    # Certificate password
    $CertificatePassword = ConvertTo-SecureString -String "secure" -AsPlainText -Force

    # Services list
    $ServicesList = @(
        'Control Panel Integration Tools',
        'Control Panel Update Tools',
        'ControlPanelIntegrationTools'
    )


### INSTALL/UPDATE 7-ZIP #######################################################

    $ZipSource      = 'https://7-zip.org/a/7z2301-x64.exe'
    $ZipDestination = Join-Path $env:TEMP '7z2301-x64.msi'
    Get-File $ZipSource $ZipDestination
    Install-File $ZipDestination

    # Alias application
    Set-Alias -scope "global" -name "7zip" -value "$env:ProgramFiles\7-Zip\7z.exe"


### INSTALL/UPDATE VCREDIST ####################################################

    $RuntimeSource        = "https://aka.ms/vs/17/release/vc_redist.x86.exe"
    $RuntimeDestination   = Join-Path $env:TEMP "vcredist_x86.exe"
    Get-File $RuntimeSource $RuntimeDestination
    Install-File $RuntimeDestination

    $Runtime64Source      = "https://aka.ms/vs/17/release/vc_redist.x64.exe"
    $Runtime64Destination = Join-Path $env:TEMP "vcredist_x64.exe"
    Get-File $RuntimeSource $RuntimeDestination
    Install-File $RuntimeDestination


### AND AWAY WE GO... ##########################################################

    # Initialize install directory
    Init-Path $InstallDirectory

    # Download files
    Get-File $FileBundleSource $DownloadDirectory

    # Extract bundle files
    Unzip-File $DownloadDirectory $InstallDirectory
    $RenameDirectory = (Get-ChildItem -Path $InstallDirectory -Filter "wcp-client-certificate-fix-files-main*" -Directory).FullName
    Rename-Item -Path $RenameDirectory -NewName $FileBundleName

    # Extract encryptor files
    Unzip-File "$($BundleDirectory)\cpit_encrypt.zip" "$($BundleDirectory)"

    # Stop Services
    foreach($Service in $ServicesList) {
        $ThisService = Get-Service -Name $Service -ErrorAction SilentlyContinue
        if ($ThisService.Length -gt 0) {
            Stop-Service -Force -DisplayName "$($Service)"
            Write-Host "`nStopping $($Service)";
        }
    }

    # Install certificate
    Write-Host "`nInstalling certificate"
    Import-PfxCertificate -InformationAction Ignore -Exportable -Password $CertificatePassword -CertStoreLocation $CertificatePath -FilePath "$($BundleDirectory)/cpit_certificate.pfx"
    Write-Host "Install complete" -ForegroundColor Green

    # Generate config
    Write-Host "`nGenerating configs"
    $proc = Start-Process -FilePath "cmd.exe" -ArgumentList "/c $($EncryptorPath) `"$($CpitDirectory)\Application.dat`" >> `"$($CpitDirectory)\Application.json`"" -PassThru -Verb RunAs
    Start-Sleep -Seconds 2
    $wshell = New-Object -ComObject Wscript.Shell
    $wshell.AppActivate($proc.Id)
    $wshell.SendKeys("{ENTER}{ENTER}")
    $proc.waitForExit()
    $proc = Start-Process -FilePath "cmd.exe" -ArgumentList "/c $($EncryptorPath) `"$($CputDirectory)\Application.dat`" >> `"$($CputDirectory)\Application.json`"" -PassThru -Verb RunAs
    Start-Sleep -Seconds 2
    $wshell = New-Object -ComObject Wscript.Shell
    $wshell.AppActivate($proc.Id)
    $wshell.SendKeys("{ENTER}{ENTER}")
    $proc.waitForExit()
    Write-Host "Generation complete" -ForegroundColor Green

    # Edit configs
    $Pattern     = '("CertificateThumbprint": (null|"[A-Z0-9]+"))'
    $Replacement = "`"CertificateThumbprint`": `"$($CertificateThumbprint)`""

    $ConfigList = @(
        "$($CpitDirectory)/Application.json",
        "$($CputDirectory)/Application.json"
    )

    Foreach($File in $ConfigList) {
        $FileContents = Get-Content -Path $File -Raw
        $NewFileContents = $FileContents -Replace $Pattern, $Replacement
        $NewFileContents | Set-Content -Path $File
    }

    # Backup databases
    Rename-Item -Path "$($CpitDirectory)\Application.dat" -NewName "$($CpitDirectory)\Application.dat_BAK"
    Rename-Item -Path "$($CputDirectory)\Application.dat" -NewName "$($CputDirectory)\Application.dat_BAK"

    # Restart Services
    foreach($Service in $ServicesList) {
        $ThisService = Get-Service -Name $Service -ErrorAction SilentlyContinue
        if ($ThisService.Length -gt 0) {
            Start-Service -DisplayName "$($Service)"
            Write-Host "`nStarting $($Service)";
        }
    }

    # Cleanup
    Remove-Item -Force -Recurse "$($InstallDirectory)/cpit_bundle"
    Remove-Item -Force "$($InstallDirectory)/$($FileBundleName).zip"

    # That's all folks!
    Write-Host "`nThat's all folks!" -ForegroundColor Yellow
